/*
 * Copyright (c) 2009, 2017, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */
package jdk.vm.ci.hotspot;

import static jdk.vm.ci.hotspot.HotSpotResolvedObjectTypeImpl.fromObjectClass;

import java.lang.invoke.CallSite;
import java.lang.invoke.ConstantCallSite;
import java.lang.invoke.MethodHandle;

import jdk.vm.ci.meta.Assumptions;
import jdk.vm.ci.meta.Constant;
import jdk.vm.ci.meta.JavaConstant;
import jdk.vm.ci.meta.JavaKind;
import jdk.vm.ci.meta.ResolvedJavaType;

/**
 * Represents a constant non-{@code null} object reference, within the compiler and across the
 * compiler/runtime interface.
 */
class HotSpotObjectConstantImpl implements HotSpotObjectConstant {

    static JavaConstant forObject(Object object) {
        return forObject(object, false);
    }

    static JavaConstant forObject(Object object, boolean compressed) {
        if (object == null) {
            return compressed ? HotSpotCompressedNullConstant.COMPRESSED_NULL : JavaConstant.NULL_POINTER;
        } else {
            return new HotSpotObjectConstantImpl(object, compressed);
        }
    }

    public static JavaConstant forBoxedValue(JavaKind kind, Object value) {
        if (kind == JavaKind.Object) {
            return HotSpotObjectConstantImpl.forObject(value);
        } else {
            return JavaConstant.forBoxedPrimitive(value);
        }
    }

    static Object asBoxedValue(Constant constant) {
        if (JavaConstant.isNull(constant)) {
            return null;
        } else if (constant instanceof HotSpotObjectConstantImpl) {
            return ((HotSpotObjectConstantImpl) constant).object;
        } else {
            return ((JavaConstant) constant).asBoxedPrimitive();
        }
    }

    private final Object object;
    private final boolean compressed;

    protected HotSpotObjectConstantImpl(Object object, boolean compressed) {
        this.object = object;
        this.compressed = compressed;
        assert object != null;
    }

    @Override
    public JavaKind getJavaKind() {
        return JavaKind.Object;
    }

    /**
     * Package-private accessor for the object represented by this constant.
     */
    Object object() {
        return object;
    }

    @Override
    public boolean isCompressed() {
        return compressed;
    }

    @Override
    public JavaConstant compress() {
        assert !compressed;
        return new HotSpotObjectConstantImpl(object, true);
    }

    @Override
    public JavaConstant uncompress() {
        assert compressed;
        return new HotSpotObjectConstantImpl(object, false);
    }

    @Override
    public HotSpotResolvedObjectType getType() {
        return fromObjectClass(object.getClass());
    }

    @Override
    public int getIdentityHashCode() {
        return System.identityHashCode(object);
    }

    @Override
    public JavaConstant getCallSiteTarget(Assumptions assumptions) {
        if (object instanceof CallSite) {
            CallSite callSite = (CallSite) object;
            MethodHandle target = callSite.getTarget();
            JavaConstant targetConstant = HotSpotObjectConstantImpl.forObject(target);
            if (!(callSite instanceof ConstantCallSite)) {
                if (assumptions == null) {
                    return null;
                }
                assumptions.record(new Assumptions.CallSiteTargetValue(this, targetConstant));
            }

            return targetConstant;
        }
        return null;
    }

    @Override
    @SuppressFBWarnings(value = "ES_COMPARING_STRINGS_WITH_EQ", justification = "reference equality is what we want")
    public boolean isInternedString() {
        if (object instanceof String) {
            String s = (String) object;
            return s.intern() == s;
        }
        return false;
    }

    @Override
    public <T> T asObject(Class<T> type) {
        if (type.isInstance(object)) {
            return type.cast(object);
        }
        return null;
    }

    @Override
    public Object asObject(ResolvedJavaType type) {
        if (type.isInstance(this)) {
            return object;
        }
        return null;
    }

    @Override
    public boolean isNull() {
        return false;
    }

    @Override
    public boolean isDefaultForKind() {
        return false;
    }

    @Override
    public Object asBoxedPrimitive() {
        throw new IllegalArgumentException();
    }

    @Override
    public int asInt() {
        throw new IllegalArgumentException();
    }

    @Override
    public boolean asBoolean() {
        throw new IllegalArgumentException();
    }

    @Override
    public long asLong() {
        throw new IllegalArgumentException();
    }

    @Override
    public float asFloat() {
        throw new IllegalArgumentException();
    }

    @Override
    public double asDouble() {
        throw new IllegalArgumentException();
    }

    @Override
    public int hashCode() {
        return System.identityHashCode(object);
    }

    @Override
    public boolean equals(Object o) {
        if (o == this) {
            return true;
        } else if (o instanceof HotSpotObjectConstantImpl) {
            HotSpotObjectConstantImpl other = (HotSpotObjectConstantImpl) o;
            return object == other.object && compressed == other.compressed;
        }
        return false;
    }

    @Override
    public String toValueString() {
        if (object instanceof String) {
            return "\"" + (String) object + "\"";
        } else {
            return JavaKind.Object.format(object);
        }
    }

    @Override
    public String toString() {
        return (compressed ? "NarrowOop" : getJavaKind().getJavaName()) + "[" + JavaKind.Object.format(object) + "]";
    }
}
