/*
 * Copyright (c) 2022, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package jdk.javadoc.internal.doclets.formats.html;

import jdk.javadoc.internal.doclets.formats.html.markup.BodyContents;
import jdk.javadoc.internal.doclets.formats.html.markup.ContentBuilder;
import jdk.javadoc.internal.doclets.formats.html.markup.HtmlAttr;
import jdk.javadoc.internal.doclets.formats.html.markup.HtmlId;
import jdk.javadoc.internal.doclets.formats.html.markup.HtmlStyle;
import jdk.javadoc.internal.doclets.formats.html.markup.TagName;
import jdk.javadoc.internal.doclets.formats.html.markup.HtmlTree;
import jdk.javadoc.internal.doclets.formats.html.Navigation.PageMode;
import jdk.javadoc.internal.doclets.formats.html.markup.Text;
import jdk.javadoc.internal.doclets.toolkit.Content;
import jdk.javadoc.internal.doclets.toolkit.util.DocFileIOException;
import jdk.javadoc.internal.doclets.toolkit.util.DocPath;
import jdk.javadoc.internal.doclets.toolkit.util.DocPaths;

/**
 * Generates the search landing page for the generated API documentation.
 */
public class SearchWriter extends HtmlDocletWriter {

    /**
     * Constructor to construct SearchWriter object.
     * @param configuration the configuration
     * @param filename file to be generated
     */
    public SearchWriter(HtmlConfiguration configuration, DocPath filename) {
        super(configuration, filename);
    }

    /**
     * Constructs the SearchWriter object and then use it to generate the search
     * file. The name of the generated file is "search.html". The search file
     * will get generated if and only if "-noindex" is not used on the command line.
     *
     * @param configuration the configuration
     * @throws DocFileIOException if there is a problem while generating the documentation
     */
    public static void generate(HtmlConfiguration configuration) throws DocFileIOException {
        DocPath filename = DocPaths.SEARCH_PAGE;
        SearchWriter searchWriter = new SearchWriter(configuration, filename);
        searchWriter.generateSearchFile();
    }

    /**
     * Generates the search file contents.
     *
     * @throws DocFileIOException if there is a problem while generating the documentation
     */
    protected void generateSearchFile() throws DocFileIOException {
        String title = resources.getText("doclet.Window_Search_title");
        HtmlTree body = getBody(getWindowTitle(title));
        ContentBuilder searchFileContent = new ContentBuilder();
        addSearchFileContents(searchFileContent);
        body.add(new BodyContents()
                .setHeader(getHeader(PageMode.SEARCH))
                .addMainContent(searchFileContent)
                .setFooter(getFooter()));
        printHtmlDocument(null, "search", body);
    }

    /**
     * Adds the search file contents to the content tree.
     */
    protected void addSearchFileContents(Content contentTree) {

        String copyText = resources.getText("doclet.Copy_url_to_clipboard");
        String copiedText = resources.getText("doclet.Copied_url_to_clipboard");
        Content helpSection = Text.EMPTY;
        // Suppress link to help page if no help page is generated or a custom help page is used.
        HtmlOptions options = configuration.getOptions();
        if (!options.noHelp() && options.helpFile().isEmpty()) {
            Content helpLink = HtmlTree.A("help-doc.html#search", contents.getContent("doclet.search.help_page_link"));
            helpSection = HtmlTree.P(contents.getContent("doclet.search.help_page_info", helpLink));
        }

        contentTree.add(HtmlTree.HEADING(Headings.PAGE_TITLE_HEADING, HtmlStyle.title,
                        contents.getContent("doclet.search.main_heading")))
                .add(HtmlTree.DIV(HtmlTree.INPUT("text", HtmlId.of("page-search-input"))
                                .put(HtmlAttr.PLACEHOLDER, resources.getText("doclet.search_placeholder")))
                        .add(HtmlTree.INPUT("reset", HtmlId.of("page-search-reset"))
                                .put(HtmlAttr.VALUE, resources.getText("doclet.search_reset"))
                                .put(HtmlAttr.STYLE, "margin: 6px;"))
                        .add(HtmlTree.DETAILS(HtmlStyle.pageSearchDetails)
                                .add(HtmlTree.SUMMARY(contents.getContent("doclet.search.show_more"))
                                        .setId(HtmlId.of("page-search-expand")))))
                .add(HtmlTree.DIV(HtmlStyle.pageSearchInfo, helpSection)
                        .add(HtmlTree.P(contents.getContent("doclet.search.keyboard_info")))
                        .add(HtmlTree.P(contents.getContent("doclet.search.browser_info")))
                        .add(HtmlTree.SPAN(Text.of("link"))
                                .setId(HtmlId.of("page-search-link")))
                        .add(new HtmlTree(TagName.BUTTON)
                                .add(new HtmlTree(TagName.IMG)
                                        .put(HtmlAttr.SRC, pathToRoot.resolve(DocPaths.CLIPBOARD_SVG).getPath())
                                        .put(HtmlAttr.ALT, copyText))
                                .add(HtmlTree.SPAN(Text.of(copyText))
                                        .put(HtmlAttr.DATA_COPIED, copiedText))
                                .addStyle(HtmlStyle.copy)
                                .setId(HtmlId.of("page-search-copy")))
                        .add(HtmlTree.P(HtmlTree.INPUT("checkbox", HtmlId.of("search-redirect")))
                                .add(HtmlTree.LABEL("search-redirect",
                                        contents.getContent("doclet.search.redirect")))))
                .add(new HtmlTree(TagName.P)
                        .setId(HtmlId.of("page-search-notify"))
                        .add(contents.getContent("doclet.search.loading")))
                .add(HtmlTree.DIV(new HtmlTree(TagName.DIV)
                                .setId(HtmlId.of("result-container"))
                                .addUnchecked(Text.EMPTY))
                        .setId(HtmlId.of("result-section"))
                        .put(HtmlAttr.STYLE, "display: none;")
                        .add(HtmlTree.SCRIPT(pathToRoot.resolve(DocPaths.SEARCH_PAGE_JS).getPath())));
    }
}
