/*
 * Copyright (c) 2003, 2021, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package jdk.javadoc.internal.doclets.toolkit.builders;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import javax.lang.model.element.Element;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.TypeElement;

import com.sun.source.doctree.DocCommentTree;
import com.sun.source.doctree.DocTree;
import jdk.javadoc.internal.doclets.toolkit.BaseOptions;
import jdk.javadoc.internal.doclets.toolkit.CommentUtils;
import jdk.javadoc.internal.doclets.toolkit.Content;
import jdk.javadoc.internal.doclets.toolkit.DocletException;
import jdk.javadoc.internal.doclets.toolkit.PropertyWriter;

import static jdk.javadoc.internal.doclets.toolkit.util.VisibleMemberTable.Kind.PROPERTIES;

/**
 * Builds documentation for a property.
 *
 *  <p><b>This is NOT part of any supported API.
 *  If you write code that depends on this, you do so at your own risk.
 *  This code and its internal interfaces are subject to change or
 *  deletion without notice.</b>
 */
public class PropertyBuilder extends AbstractMemberBuilder {

    /**
     * The writer to output the property documentation.
     */
    private final PropertyWriter writer;

    /**
     * The list of properties being documented.
     */
    private final List<? extends Element> properties;

    /**
     * The index of the current property that is being documented at this point
     * in time.
     */
    private ExecutableElement currentProperty;

    /**
     * Construct a new PropertyBuilder.
     *
     * @param context  the build context.
     * @param typeElement the class whose members are being documented.
     * @param writer the doclet specific writer.
     */
    private PropertyBuilder(Context context,
            TypeElement typeElement,
            PropertyWriter writer) {
        super(context, typeElement);
        this.writer = Objects.requireNonNull(writer);
        properties = getVisibleMembers(PROPERTIES);
    }

    /**
     * Construct a new PropertyBuilder.
     *
     * @param context  the build context.
     * @param typeElement the class whose members are being documented.
     * @param writer the doclet specific writer.
     * @return the new PropertyBuilder
     */
    public static PropertyBuilder getInstance(Context context,
            TypeElement typeElement,
            PropertyWriter writer) {
        return new PropertyBuilder(context, typeElement, writer);
    }

    /**
     * Returns whether or not there are members to document.
     *
     * @return whether or not there are members to document
     */
    @Override
    public boolean hasMembersToDocument() {
        return !properties.isEmpty();
    }

    @Override
    public void build(Content contentTree) throws DocletException {
        buildPropertyDoc(contentTree);
    }

    /**
     * Build the property documentation.
     *
     * @param detailsList the content tree to which the documentation will be added
     * @throws DocletException if there is a problem while building the documentation
     */
    protected void buildPropertyDoc(Content detailsList) throws DocletException {
        if (hasMembersToDocument()) {
            Content propertyDetailsTreeHeader = writer.getPropertyDetailsTreeHeader(detailsList);
            Content memberList = writer.getMemberList();

            for (Element property : properties) {
                currentProperty = (ExecutableElement)property;
                Content propertyDocTree = writer.getPropertyDocTreeHeader(currentProperty);

                buildSignature(propertyDocTree);
                buildPropertyComments(propertyDocTree);
                buildTagInfo(propertyDocTree);

                memberList.add(writer.getMemberListItem(propertyDocTree));
            }
            Content propertyDetails = writer.getPropertyDetails(propertyDetailsTreeHeader, memberList);
            detailsList.add(propertyDetails);
        }
    }

    /**
     * Build the signature.
     *
     * @param propertyDocTree the content tree to which the documentation will be added
     */
    protected void buildSignature(Content propertyDocTree) {
        propertyDocTree.add(writer.getSignature(currentProperty));
    }

    /**
     * Build the deprecation information.
     *
     * @param propertyDocTree the content tree to which the documentation will be added
     */
    protected void buildDeprecationInfo(Content propertyDocTree) {
        writer.addDeprecated(currentProperty, propertyDocTree);
    }

    /**
     * Build the preview information.
     *
     * @param propertyDocTree the content tree to which the documentation will be added
     */
    protected void buildPreviewInfo(Content propertyDocTree) {
        writer.addPreview(currentProperty, propertyDocTree);
    }

    /**
     * Build the comments for the property.  Do nothing if
     * {@link BaseOptions#noComment()} is set to true.
     *
     * @param propertyDocTree the content tree to which the documentation will be added
     */
    protected void buildPropertyComments(Content propertyDocTree) {
        if (!options.noComment()) {
            writer.addComments(currentProperty, propertyDocTree);
        }
    }

    /**
     * Build the tag information.
     *
     * @param propertyDocTree the content tree to which the documentation will be added
     */
    protected void buildTagInfo(Content propertyDocTree) {
        CommentUtils cmtUtils = configuration.cmtUtils;
        DocCommentTree dct = utils.getDocCommentTree(currentProperty);
        var fullBody = dct.getFullBody();
        ArrayList<DocTree> blockTags = dct.getBlockTags().stream()
                .filter(t -> t.getKind() != DocTree.Kind.RETURN)
                .collect(Collectors.toCollection(ArrayList::new));
        String sig = "#" + currentProperty.getSimpleName() + "()";
        blockTags.add(cmtUtils.makeSeeTree(sig, currentProperty));
        // The property method is used as a proxy for the property
        // (which does not have an explicit element of its own.)
        // Temporarily override the doc comment for the property method
        // by removing the `@return` tag, which should not be displayed for
        // the property.
        CommentUtils.DocCommentInfo prev = cmtUtils.setDocCommentTree(currentProperty, fullBody, blockTags);
        try {
            writer.addTags(currentProperty, propertyDocTree);
        } finally {
            cmtUtils.setDocCommentInfo(currentProperty, prev);
        }
    }

    /**
     * Return the property writer for this builder.
     *
     * @return the property writer for this builder.
     */
    public PropertyWriter getWriter() {
        return writer;
    }
}
