/*
 * Copyright (c) 2018, 2020, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */


package org.graalvm.compiler.replacements.amd64;

import static org.graalvm.compiler.nodeinfo.InputType.Memory;
import static org.graalvm.compiler.nodeinfo.NodeCycles.CYCLES_UNKNOWN;
import static org.graalvm.compiler.nodeinfo.NodeSize.SIZE_512;

import org.graalvm.compiler.core.common.type.StampFactory;
import org.graalvm.compiler.graph.NodeClass;
import org.graalvm.compiler.lir.gen.LIRGeneratorTool;
import org.graalvm.compiler.nodeinfo.NodeInfo;
import org.graalvm.compiler.nodes.FixedWithNextNode;
import org.graalvm.compiler.nodes.NamedLocationIdentity;
import org.graalvm.compiler.nodes.ValueNode;
import org.graalvm.compiler.nodes.ValueNodeUtil;
import org.graalvm.compiler.nodes.memory.MemoryAccess;
import org.graalvm.compiler.nodes.memory.MemoryKill;
import org.graalvm.compiler.nodes.memory.MultiMemoryKill;
import org.graalvm.compiler.nodes.spi.LIRLowerable;
import org.graalvm.compiler.nodes.spi.NodeLIRBuilderTool;
import jdk.internal.vm.compiler.word.LocationIdentity;
import jdk.internal.vm.compiler.word.Pointer;

import jdk.vm.ci.meta.JavaKind;
import jdk.vm.ci.meta.Value;

@NodeInfo(allowedUsageTypes = Memory, size = SIZE_512, cycles = CYCLES_UNKNOWN)

public final class AMD64StringUTF16CompressNode extends FixedWithNextNode
                implements LIRLowerable, MultiMemoryKill, MemoryAccess {

    public static final NodeClass<AMD64StringUTF16CompressNode> TYPE = NodeClass.create(AMD64StringUTF16CompressNode.class);

    @Input private ValueNode src;
    @Input private ValueNode dst;
    @Input private ValueNode len;
    final JavaKind readKind;

    @OptionalInput(Memory) private MemoryKill lla; // Last access location registered.

    // java.lang.StringUTF16.compress([CI[BII)I
    //
    // int compress(char[] src, int src_indx, byte[] dst, int dst_indx, int len)
    //
    // Represented as a graph node by:

    public AMD64StringUTF16CompressNode(ValueNode src, ValueNode dst, ValueNode len, JavaKind readKind) {
        super(TYPE, StampFactory.forInteger(32));
        this.src = src;
        this.dst = dst;
        this.len = len;
        this.readKind = readKind;
    }

    @Override
    public LocationIdentity getLocationIdentity() {
        // Model read access via 'src' using:
        return NamedLocationIdentity.getArrayLocation(readKind);
    }

    @Override
    public LocationIdentity[] getKilledLocationIdentities() {
        // Model write access via 'dst' using:
        return new LocationIdentity[]{NamedLocationIdentity.getArrayLocation(JavaKind.Byte)};
    }

    @Override
    public void generate(NodeLIRBuilderTool gen) {
        LIRGeneratorTool lgt = gen.getLIRGeneratorTool();
        Value res = lgt.emitStringUTF16Compress(gen.operand(src), gen.operand(dst), gen.operand(len));
        gen.setResult(this, res);
    }

    @Override
    public MemoryKill getLastLocationAccess() {
        return lla;
    }

    @Override
    public void setLastLocationAccess(MemoryKill newlla) {
        updateUsages(ValueNodeUtil.asNode(lla), ValueNodeUtil.asNode(newlla));
        lla = newlla;
    }

    @NodeIntrinsic
    public static native int compress(Pointer src, Pointer dst, int len, @ConstantNodeParameter JavaKind readKind);
}
